/*******************************************************************************
* CMUcam4 Interface Library Source
*
* Author: Christopher J. Leaf
* Updated: 3/20/2012
* Version: 0.1
*
* Copyright (c) 2012 Christopher J. Leaf
* See end of file for terms of use.
*
* Update History:
*
* v0.1 - Original beta release - 03/20/2012
*******************************************************************************/

#include "CMUCam4.h"
#include "WProgram.h"
#include "HardwareSerial.h"
#include <setjmp.h>
#include <String.h>

int CMUCam4::init(){
	int err;
	int version = 0;
	
	Serial.begin(CMUCAM4_BAUD_RATE);
	_setState(CMUCAM4_VOID_STATE);
	reset();
	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return CMUCAM4_RETURN_FAIL;
	}

	_serialRead(); //Clear Version String

	_waitIdle();

	return version == 1 ? CMUCAM4_RETURN_SUCCESS : CMUCAM4_RETURN_FAIL;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::idle(){
	int err;

	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("");
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::getVersionNumber(char* versionString, int size){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("GV");
	_waitACK();
	_serialRead();
	if(  strncmp(_byteBuffer, "CMUcam4 v1.00", 8) != 0  ){
		THROW_ERROR(CMUCAM4_ERR_BAD_STM);
	}
	_waitIdle();

	strncpy(versionString, _byteBuffer, size);

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::reset(){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_setState(CMUCAM4_VOID_STATE);

	_sendCommand("RS");
	_waitACK();
	delay(2500);	//Give time to complete resetting completely

	_serialRead();
	if(  strncmp(_byteBuffer, "CMUcam4 v1.00", 13) == 0  ){
		THROW_ERROR(CMUCAM4_ERR_BAD_STM);
	}

	_waitIdle();
	return err;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::startSleepDeeply(){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		_unsetState(CMUCAM4_SLEEPING);
		return err;
	}

	_sendCommand("SD");
	_waitACK();	
	delay(60);	//Give enough time to ensure it is fully asleep
	_setState(CMUCAM4_SLEEPING);

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::startSleepLightly(){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		_unsetState(CMUCAM4_SLEEPING);
		return err;
	}

	_sendCommand("SL");
	_waitACK();
	_setState(CMUCAM4_SLEEPING);	

	return CMUCAM4_RETURN_SUCCESS; 
}

CMUCam4::CMUCAM4_ERROR CMUCam4::wakeup(){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		_setState(CMUCAM4_SLEEPING);
		return err;	
	}
	_unsetState(CMUCAM4_SLEEPING);
	_sendCommand("");	//Just needs to send something
	_waitACK();
	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setCameraBrightness(int brightness){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"CB %d", brightness & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setCameraContrast(int contrast){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"CC %d", contrast & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setAutoGain(int active){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"AG %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setAutoWhiteBalance(int active){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"MF %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setHorizontalMirrorMode(int active){	
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"HM %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}	

CMUCam4::CMUCAM4_ERROR CMUCam4::setVerticalFlipMode(int active){
	int err;	
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"VF %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}	

CMUCam4::CMUCAM4_ERROR CMUCam4::setBlackWhiteMode(int active){	
	int err;	
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"BW %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}	

CMUCam4::CMUCAM4_ERROR CMUCam4::setNegativeMode(int active){	
	int err;	
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"NG %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}	

int CMUCam4::getButtonState(){
	int err;
	int returnValue;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("GB");
	_waitACK();
	returnValue = _readInt(NULL);
	_waitIdle();
		
	return returnValue;
}

int CMUCam4::getButtonDuration(){
	int err;
	int returnValue;
	Serial.flush();
	if( 0 != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("GD");
	_waitACK();
	returnValue = _readInt(NULL);
	_waitIdle();
		
	return returnValue;
}

int CMUCam4::getButtonPressed(){
	int err;
	int returnValue;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("GP");
	_waitACK();
	returnValue = _readInt(NULL);
	_waitIdle();
		
	return returnValue;
}

int CMUCam4::getButtonReleased(){
	int err;
	int returnValue;
	Serial.flush();

	if( 0 != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("GR");
	_waitACK();
	returnValue = _readInt(NULL);
	_waitIdle();
		
	return returnValue;
}

int CMUCam4::panInput(){
	int err;
	int returnValue;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("PI");
	_waitACK();
	returnValue = _readInt(NULL);
	_waitIdle();

	return returnValue;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::panOutput(int direction, int output){	
	int err;	
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE, 
					"PO %d %d", direction & 0xFF, output & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

int CMUCam4::tiltInput(){
	int err;
	int returnValue;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("TI");
	_waitACK();
	returnValue = _readInt(NULL);
	_waitIdle();

	return returnValue;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::tiltOutput(int direction, int output){	
	int err;	
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE, 
		"TO %d %d", direction & 0xFF, output & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::getInputs(){
	int err;
	int returnValue;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("GI");
	_waitACK();
	returnValue = _readInt(NULL);
	_waitIdle();
	return returnValue;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setOutputs(int direction, int output){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return CMUCAM4_RETURN_FAIL;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
		"TO %d %d", direction & 0xFF, output & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::turnOffLED(){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("L0");
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::turnOnLED(int frequency){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"L1 %d", frequency & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

int CMUCam4::getServoPosition(int servo){
	int err;
	int returnValue;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"GS %d", servo & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	returnValue = _readInt(NULL);
	_waitIdle();
		
	return returnValue;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setServoState
						(int servo, int enable, int pulse){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"SS %d %d %d", servo & 0xFF, enable & 0xFF, pulse & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::autoTilt(int active, int isReversed){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}
	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"AT %d %d", active & 0xFF, isReversed & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::autoPan(int active, int isReversed){
	int err;	
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"AP %d %d", active & 0xFF, isReversed & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setAutoPanParameters
						(int proportional, int derivative){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"PP %d %d", proportional & 0xFF, derivative & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setAutoTiltParameters
						(int proportional, int derivative){	
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"TP %d %d", proportional & 0xFF, derivative & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::turnOnMonitor(){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("M1");
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::turnOffMonitor(){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("M0");
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::freezeMonitor(int active){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"MF %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::monitorSignal(int signal){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"MS %d", signal & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::getTrackParameters
						(CMUCAM4_color_range_t * colors){
	int err;
	unsigned char read;	
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("GT");
	_waitACK();

	colors->redLow = _readInt(NULL);
	colors->redHigh = _readInt(NULL);
	colors->blueLow = _readInt(NULL);
	colors->blueHigh = _readInt(NULL);
	colors->greenLow = _readInt(NULL);
	colors->greenHigh = _readInt(NULL);
		
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::getTrackWindow
						(CMUCAM4_track_window_t * trkWindow){
	int err;
	Serial.flush();

	if(NULL == trkWindow)
		return CMUCAM4_RETURN_FAIL;

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	_sendCommand("GW");
	_waitACK();

	trkWindow->boundX1 = _readInt(NULL);
	trkWindow->boundY1 = _readInt(NULL);
	trkWindow->boundX2 = _readInt(NULL);
	trkWindow->boundY2 = _readInt(NULL);

	_waitIdle();
	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setTrackColor(CMUCAM4_color_range_t * colors){
	if(NULL == colors){
		return CMUCAM4_RETURN_FAIL;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
		"ST %d %d %d %d %d %d",  colors->redLow, colors->redHigh,
		colors->blueLow, colors->blueHigh,
		colors->greenLow, colors->greenHigh);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setTrackWindow 
						(CMUCAM4_track_window_t * trkWindow){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}
	
	if(NULL == trkWindow){
		snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE, "SW %d %d %d %d", 
			CMUCAM4_WINDOW_X_MIN, CMUCAM4_WINDOW_Y_MIN, 
			CMUCAM4_WINDOW_X_MAX, CMUCAM4_WINDOW_Y_MAX);
	}
	else{
		snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE, "SW %d %d %d %d", 
			trkWindow->boundX1, trkWindow->boundY1, 
			trkWindow->boundX2, trkWindow->boundY2);
	}
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();
	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::startTrackColor(CMUCAM4_color_range_t * colors){
	int err;		
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		_unsetState(CMUCAM4_TRACK_COLOR_STREAM);
		return err;
	}

	if(NULL == colors){
		return _sendCommand("TC");
	}else{
		snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"TC %d %d %d %d %d %d",  colors->redLow, colors->redHigh,
			colors->blueLow, colors->blueHigh,
			colors->greenLow, colors->greenHigh);
		
		_sendCommand(_commandBuffer);
	}

	_waitACK();
	_setState(CMUCAM4_TRACK_COLOR_STREAM);

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::startTrackThrsh
								(CMUCAM4_color_thrsh_t * thrsh){
	int err;
	Serial.flush();

	if(NULL == thrsh){
		return CMUCAM4_RETURN_FAIL;
	}

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		_unsetState(CMUCAM4_TRACK_COLOR_STREAM);
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE, 
		"TW %d %d %d", &thrsh->redThrsh, &thrsh->greenThrsh,&thrsh->blueThrsh);
	_sendCommand(_commandBuffer);
	_waitACK();
	_setState(CMUCAM4_TRACK_COLOR_STREAM);

	return CMUCAM4_RETURN_SUCCESS;												
}

CMUCam4::CMUCAM4_ERROR CMUCam4::getTrackData(CMUCAM4_track_data_t * data){
	int err;

	if(NULL == data){
		return CMUCAM4_RETURN_FAIL;
	}
	if(_getState() != CMUCAM4_TRACK_COLOR_STREAM){
		return CMUCAM4_ERR_STATE;
	}

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	if(_readChar() != 'T' || _readChar() != ' ')
		return CMUCAM4_ERR_BAD_STM;

	data->centroidX = _readInt(NULL);
	data->centroidY = _readInt(NULL);
	data->boundX1 = _readInt(NULL);
	data->boundY1 = _readInt(NULL);
	data->boundX2 = _readInt(NULL);
	data->boundY2 = _readInt(NULL);
	data->trackedPixels = _readInt(NULL);
	data->trackingConfidence = _readInt(NULL);

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::stopTracking(){
	_unsetState(CMUCAM4_TRACK_COLOR_STREAM);
	idle();
	return idle();
}

CMUCam4::CMUCAM4_ERROR CMUCam4::pollTrackColor
				(CMUCAM4_color_range_t * colors, CMUCAM4_track_data_t * data){
	int err;
	
	if(CMUCAM4_RETURN_SUCCESS != (err = startTrackColor(colors))){
		return err;
	}
	if(CMUCAM4_RETURN_SUCCESS != (err = getTrackData(data))){
		return err;
	}
	if(CMUCAM4_RETURN_SUCCESS != (err = stopTracking())){
		return err;
	}
	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::pollTrackThrsh
				(CMUCAM4_color_thrsh_t * colors, CMUCAM4_track_data_t * data){
	int err;
	
	if(CMUCAM4_RETURN_SUCCESS != (err = startTrackThrsh(colors))){
		return err;
	}
	if(CMUCAM4_RETURN_SUCCESS != (err = getTrackData(data))){
		return err;
	}
	if(CMUCAM4_RETURN_SUCCESS != (err = stopTracking())){
		return err;
	}
	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::startHistogram(int color, int numBins){
	int err;
	int logNumBins = 0;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		_unsetState(CMUCAM4_HISTOGRAM_STREAM);	
		return err;
	}

	//Calculate log, floor, of numBins
	while(numBins >>= 1){
		logNumBins++;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
		"GH %d %d", color & 0xFF, logNumBins & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_setState(CMUCAM4_HISTOGRAM_STREAM);	

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::getHistogramData
							(unsigned char * bins, int numBins){
	int err;
	int numRead = 0;
	char delimiter = '\0';

	if(NULL == bins){
		return CMUCAM4_RETURN_FAIL;
	}
	if(_getState() != CMUCAM4_HISTOGRAM_STREAM){
		return CMUCAM4_ERR_STATE;
	}

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	if(_readChar() != 'H' || _readChar() != ' '){
		return CMUCAM4_ERR_BAD_STM;
	}

	while(delimiter != CMUCAM4_ENDLINE_CHAR && numRead < numBins){
		bins[numRead] = _readInt(&delimiter);
		numRead++;
	}

	return numRead;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::stopHistogram(){
	_unsetState(CMUCAM4_HISTOGRAM_STREAM);	
	idle();
	return idle();
}

CMUCam4::CMUCAM4_ERROR CMUCam4::pollHistogram
							(int color, unsigned char * bins, int numBins){
	int err;
	if(CMUCAM4_RETURN_SUCCESS != (err = startHistogram(color, numBins))){
		return err;
	}
	if(CMUCAM4_RETURN_SUCCESS != (err = getHistogramData(bins, numBins))){
		return err;
	}
	if(CMUCAM4_RETURN_SUCCESS != (err = stopHistogram())){
		return err;
	}
	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::startMean(){
	int err;
	int returnValue;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		_unsetState(CMUCAM4_GET_MEAN_STREAM);	
		return err;
	}

	_sendCommand("GM");
	_waitACK();
	_setState(CMUCAM4_GET_MEAN_STREAM);	

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::getMean(CMUCam4::CMUCAM4_mean_data_t * data){
	int err;

	if(NULL == data){
		return CMUCAM4_RETURN_FAIL;
	}
	if(_getState() != CMUCAM4_GET_MEAN_STREAM){
		return CMUCAM4_ERR_STATE;
	}

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	if(_readChar() != 'S' || _readChar() != ' ')
		return CMUCAM4_ERR_BAD_STM;

	data->redMean = _readInt(NULL);
	data->greenMean = _readInt(NULL);
	data->blueMean = _readInt(NULL);
	data->redMedian = _readInt(NULL);
	data->greenMedian = _readInt(NULL);
	data->blueMedian = _readInt(NULL);
	data->redMode = _readInt(NULL);
	data->greenMode = _readInt(NULL);
	data->blueMode = _readInt(NULL);
	data->redStd = _readInt(NULL);
	data->greenStd = _readInt(NULL);
	data->blueStd = _readInt(NULL);

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::stopMean(){
	_unsetState(CMUCAM4_GET_MEAN_STREAM);	
	idle();
	return idle();
}

CMUCam4::CMUCAM4_ERROR CMUCam4::pollMean(CMUCam4::CMUCAM4_mean_data_t * data){
	int err;
	if(CMUCAM4_RETURN_SUCCESS != (err = startMean())){
		return err;
	}
	if(CMUCAM4_RETURN_SUCCESS != (err = getMean(data))){
		return err;
	}
	if(CMUCAM4_RETURN_SUCCESS != (err = stopMean())){
		return err;
	}
	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::startTestMode(int active){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"TM %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setColorTracking(int active){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
		"CT %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setHistogramTracking(int active){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
		"HT %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}

CMUCam4::CMUCAM4_ERROR CMUCam4::setInvertedFilterMode(int active){
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}

	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"NF %d", active & 0xFF);
	_sendCommand(_commandBuffer);
	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}	

CMUCam4::CMUCAM4_ERROR CMUCam4::setNoiseFilterMode(int thresh){	
	int err;
	Serial.flush();

	if( CMUCAM4_RETURN_SUCCESS != (err = CATCH_ERROR()) ){
		return err;
	}
	snprintf(_commandBuffer, CMUCAM4_MAX_COMMAND_SIZE,
			"NF %d", thresh & 0xFF);
	_sendCommand(_commandBuffer);

	_waitACK();
	_waitIdle();

	return CMUCAM4_RETURN_SUCCESS;
}	

int CMUCam4::_setState(int state){
	if(_state != CMUCAM4_VOID_STATE && state != CMUCAM4_VOID_STATE){
		THROW_ERROR(CMUCAM4_ERR_STATE);
	}
	_state = state;
	return CMUCAM4_RETURN_SUCCESS;
}

int CMUCam4::_unsetState(int state){
	if(_state != CMUCAM4_VOID_STATE && _state != state){
		THROW_ERROR(CMUCAM4_ERR_STATE);
	}
	_state = CMUCAM4_VOID_STATE;
	return CMUCAM4_RETURN_SUCCESS;
}

int CMUCam4::_getState(){
	return _state;
}

int CMUCam4::_sendCommand(const char * command){
	if(_getState() != CMUCAM4_VOID_STATE)
		THROW_ERROR(CMUCAM4_ERR_NCK);
		
	Serial.print(command);
	Serial.print(CMUCAM4_ENDLINE_CHAR);
	return CMUCAM4_RETURN_SUCCESS;
}

char CMUCam4::_readChar(){
	char _serialByte;
	unsigned long _serialTimer; 
	_serialTimer = millis();

	while(Serial.available() == 0){
		delay(10);
		if( ( millis() - _serialTimer ) > CMUCAM4_SERIAL_TIMEOUT )
			THROW_ERROR(CMUCAM4_ERR_TIMEOUT);
	}

	_serialByte = Serial.read();

	return _serialByte;
}

int CMUCam4::_readInt(char * delim){
	char localBuf[CMUCAM4_LOCAL_BUFF_SIZE];
	int byteCount = 0;
	char readInt = '\0';
	int readValue = -1;
	unsigned long _serialTimer; 
	_serialTimer = millis();

	while(byteCount < CMUCAM4_LOCAL_BUFF_SIZE){
		if(Serial.available()){
			readInt = Serial.read();
			if(readInt < '0' || readInt > '9'){
				localBuf[byteCount] = '\0';
				if(sscanf(localBuf, "%d", &readValue) != 1)
					THROW_ERROR(CMUCAM4_ERR_BAD_STM);
				if(delim)
					*delim = readInt;
				return readValue;
			}
			else{
				localBuf[byteCount] = readInt;
				byteCount++;
			}
		}
		else{
			delay(10);
			if( ( millis() - _serialTimer ) > CMUCAM4_SERIAL_TIMEOUT )
				THROW_ERROR(CMUCAM4_ERR_TIMEOUT);
		}		
	}
	return -1;
}

int CMUCam4::_waitACK(){
	while(1){
		_serialRead();

		if(strncmp(_byteBuffer, "ACK", 3) == 0)
			break;
			
		else if(strncmp(_byteBuffer, "NCK", 3) == 0){
			THROW_ERROR(CMUCAM4_ERR_NCK);
		}
	}
	return CMUCAM4_RETURN_SUCCESS;
}

int CMUCam4::_waitIdle(){
	do{
		_serialRead();
	}while( _byteBuffer[0] != ':' );

	return CMUCAM4_RETURN_SUCCESS;
}

int CMUCam4::_serialRead(){
	char _serialByte;
	unsigned long _serialTimer; 
	_serialTimer = millis();
	_byteCount = 0;

	while(Serial.available() == 0){
		delay(10);
		if( ( millis() - _serialTimer ) > CMUCAM4_SERIAL_TIMEOUT )
			THROW_ERROR(CMUCAM4_ERR_TIMEOUT);
	}
	
	_serialByte = Serial.read();
	_byteBuffer[0] = _serialByte;
	_byteCount++;

	if(_serialByte == ':')
		return CMUCAM4_RETURN_SUCCESS;

	while( _byteCount < CMUCAM4_SERIAL_SIZE){
		if(Serial.available()){
			_serialByte = Serial.read();			
			_byteBuffer[_byteCount] = _serialByte;

			if(_serialByte == CMUCAM4_ENDLINE_CHAR){ 
				_byteBuffer[_byteCount] = NULL;
				return CMUCAM4_RETURN_SUCCESS;
			}
		
			_byteCount++;
		}
		else{
			delay(10);
			if( ( millis() - _serialTimer ) > CMUCAM4_SERIAL_TIMEOUT )
				THROW_ERROR(CMUCAM4_ERR_TIMEOUT);
		}
		
		if(_byteCount == 4 && strncmp("ERR:", _byteBuffer, 4) == 0 ){
			THROW_ERROR(CMUCAM4_ERR_ERROR);
		}	
	}
	_byteBuffer[_byteCount] = NULL;

	THROW_ERROR(CMUCAM4_ERR_FLOW);
}

/*******************************************************************************
*                     TERMS OF USE: MIT License
********************************************************************************
* Permission is hereby granted, free of charge, to any person obtaining a copy 
* of this software and associated documentation files (the "Software"), to deal 
* in the Software without restriction, including without limitation the rights 
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
* copies of the Software, and to permit persons to whom the Software is 
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in 
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*******************************************************************************/
