/*******************************************************************************
* CMUcam4 Serial Interface Library Header
*
* Author: Christopher J. Leaf
* Updated: 3/20/2012
* Version: v0.1
*
* Copyright (c) 2012 Christopher J. Leaf
* See end of file for terms of use.
*
* Update History:
*
* v0.1 - Original Beta release - 3/20/2012
*******************************************************************************/

#ifndef _CMUCAM4_H_
#define _CMUCAM4_H_ 

/*******************************************************************************
* Possible compatibility fix here, if it doesn't work try removing it. Other-
* wise consult Arduino's documentation for your version of the Arduino GUI.
*******************************************************************************/
#if defined(ARDUINO) && ARDUINO >= 100
#include "Arduino.h"
#else
#include "WProgram.h"
#endif

#include "HardwareSerial.h"
#include <setjmp.h>
#include <String.h>

#define CMUCAM4_BAUD_RATE       19200
#define CMUCAM4_SERIAL_SIZE       256
#define CMUCAM4_SERIAL_TIMEOUT    500

#define CMUCAM4_MAX_COMMAND_SIZE  128
#define CMUCAM4_LOCAL_BUFF_SIZE    64
#define CMUCAM4_ENDLINE_CHAR      '\r'

#define CMUCAM4_WINDOW_X_MIN 		0
#define CMUCAM4_WINDOW_Y_MIN 		0
#define CMUCAM4_WINDOW_X_MAX 	  160 
#define CMUCAM4_WINDOW_Y_MAX 	  120

/*******************************************************************************
* Note on error handling:
* 
*   Any command wrapping function (all the public ones) should be assumed to 
* return any of the errors associated with reading from the serial line unless
* otherwise specified. All errors are negative numbers and nonnegative numbers
* refer to success. No function will 
*   This library is designed to handle catching errors at a low level (in the 
* private helper functions) and passes errors up to the public functions to be
* caught. It is vital to proper program behavior that these private functions 
* are not called directly without catching the errors they can throw. Failing 
* to catch these errors (in the style used by each command function) will result
* in possibly undefined behavior and correct program execution cannot be 
* guaranteed. Those familiar with exception handling can understand this as a 
* form of uncaught exception - the error handling is modelled by a try catch
* style of programming.
*   This library provides two wrappers for the error catching, defined below. 
* They use a buffer to store environment information in order to work, and it 
* should not directly be modified. CATCH_ERROR() needs to be called before any
* function that will call THROW_ERROR(error). The style used by this library is 
* to set up a CATCH_ERROR() before any low level call, and if it fails it 
* returns that error immediately. If the program runs into an error it will 
* return to this CATCH_ERROR() again and then return the error from the 
* function. Thus if the program is still running below where the CATCH_ERROR is 
* set, it can safely be assumed all errors (at least those handled by this 
* library) have not occured. So if the program reaches the final return 
* statement it executed without errors.
* 
*   This paragraph is a more technical explanation of how errors are implemented
* and can be skipped as the error handling is effectively hidden. Errors are 
* generated and caught using the low level construct of setjmp and longjmp. 
* setjmp essentially stores program execution information, returning 0 at first 
* call and will be called again by longjmp returning whatever nonzero value 
* is given to longjmp. It is important to realize that longjmp will cause the 
* most recent setjmp to return again, regardless of if that was intended. This
* is why catching errors is important to correct execution. For this reason
* each function that is to call at least one of the lower level functions which 
* relate directly to reading/writing to the CMUCam4 needs to use setjmp so that
* if the low level function encounters an error it can longjmp to the correct
* function body, if not it will jump to an undefined state in the program 
* execution. Further information on this can be found online. Even deeper 
* understanding requires knowledge of the program stack and assembly language. 
* A simple way to ensure controlled execution however is just to ensure that
* there is always a setjmp before any longjmp can be called and if that setjmp
* fails immediately return from the function without calling another function.
*******************************************************************************/
#define CATCH_ERROR()       setjmp(_longJumpEnv)
#define THROW_ERROR(error)  longjmp(_longJumpEnv, error)

/*******************************************************************************
* Definitions of specific return values regarding CMUCam4's execution.
* 
* Defines the types of messages CMUCam4 can return in errno (defined below).
* Errors cannot coexist but affect integral value multiple messages exist.
* Errors can be returned by each function regarding serial communication.
* RETURN_FAIL is returend by specific functions, explained where relevant.
* ERR_STATE can be returned by any function, explained where relevant.
*
* CMUCAM4_RETURN_SUCCESS     0	Command completed successfully
* CMUCAM4_RETURN_FAIL       -1	Command failed for special reason.
* CMUCAM4_ERR_TIMEOUT       -2	Command/response timed out.
* CMUCAM4_ERR_FLOW 	        -3	Buffered Serial read overflow or short count.
* CMUCAM4_ERR_ERROR         -4	Special CMUCam4 error message received
* CMUCAM4_ERR_BAD_STM       -5	CMUCam4 sent an unexpected response stream.
* CMUCAM4_ERR_NCK           -6  CMUCAM4 rejected the inputted command.	
* CMUCAM4_ERR_STATE	        -7 	Invalid command ordering, CMUCam4 is busy doing  
*                                   a special task requiring state.	
*******************************************************************************/
#define CMUCAM4_RETURN_SUCCESS   0
#define CMUCAM4_RETURN_FAIL     -1
#define CMUCAM4_ERR_TIMEOUT     -2
#define CMUCAM4_ERR_FLOW        -3
#define CMUCAM4_ERR_ERROR       -4
#define CMUCAM4_ERR_BAD_STM     -5
#define CMUCAM4_ERR_NCK         -6
#define CMUCAM4_ERR_STATE       -7

/*******************************************************************************
* Definitions of CMUCam4 of possible states it can be executing in.
* 
* If state is nonzero then _sendCommand() will return ERR_STATE error. Thus any
* function that sends data via _sendCommand() can cause this error. 
*
* Information about each state is described in the headers where it is changed.  
*******************************************************************************/
#define CMUCAM4_VOID_STATE          0
#define CMUCAM4_HISTOGRAM_STREAM    1
#define CMUCAM4_TRACK_COLOR_STREAM  2
#define CMUCAM4_SLEEPING            4
#define CMUCAM4_GET_MEAN_STREAM	    8

class CMUCam4{


/*******************************************************************************
********************************************************************************
********   PUBLIC
********************************************************************************
*******************************************************************************/
public:

/*******************************************************************************
* CMUCAM4_ERROR : Return type of command wrappers that do not return data.
*******************************************************************************/
typedef int CMUCAM4_ERROR;

/*******************************************************************************
* CMUCAM4_color_thrsh_t : Used by Tracking commands - RBG color ranges.
*******************************************************************************/
typedef struct{
	unsigned char redLow;
	unsigned char redHigh;
	unsigned char greenLow;
	unsigned char greenHigh;
	unsigned char blueLow;
	unsigned char blueHigh;
} CMUCAM4_color_range_t; 

/*******************************************************************************
* CMUCAM4_track_thrsh_t: Used by Tracking commands - Color variance. 
*******************************************************************************/
typedef struct{
	unsigned char redThrsh;
	unsigned char greenThrsh;
	unsigned char blueThrsh;
}CMUCAM4_color_thrsh_t;

/*******************************************************************************
* CMUCAM4_track_data_t : Used by Tracking commands - Tracking information.
*******************************************************************************/
typedef struct{
	unsigned char centroidX;
	unsigned char centroidY;
	unsigned char boundX1;
	unsigned char boundY1;
	unsigned char boundX2;
	unsigned char boundY2;
	unsigned char trackedPixels;
	unsigned char trackingConfidence;
} CMUCAM4_track_data_t;

/*******************************************************************************
* CMUCAM4_track_window_t : Used by Tracking commands - Defines a track window.
*******************************************************************************/
typedef struct{
	unsigned char boundX1;
	unsigned char boundY1;
	unsigned char boundX2;
	unsigned char boundY2;
} CMUCAM4_track_window_t;

/*******************************************************************************
* CMUCAM4_mean_data_t : Used by Mean commands - Statistic information. 
*******************************************************************************/
typedef struct{
	unsigned char redMean;
	unsigned char greenMean;
	unsigned char blueMean;
	unsigned char redMedian;
	unsigned char greenMedian;
	unsigned char blueMedian;
	unsigned char redMode;
	unsigned char greenMode;
	unsigned char blueMode;
	unsigned char redStd;
	unsigned char greenStd;
	unsigned char blueStd;
}CMUCAM4_mean_data_t;


/*******************************************************************************
* init()
* Starts up the CMUCam4 interface library.
*
* Must be called before using any other functions in this library.
*
* Returns CMUCAM4_RETURN_SUCCESS on success.
*******************************************************************************/	
CMUCAM4_ERROR init();		

/*******************************************************************************
* idle()
* Forcefully makes the CMUCam4 begin to idle.
*
* Interupts current operation CMUCam4 and begin waiting state.
* Cannot interrupt special states. 
* Can act as a ping, to check if CMUCam4 is currently responsive.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR idle();		

/*******************************************************************************
* getVersionNumber(char *versionString, int size)
* Get information string from CMUCam4 regarding it's version number
*
* versionString: A pointer to an allocated buffer to store result into.
* size: The size of the buffer pointed to by versionString.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure. 
*******************************************************************************/
CMUCAM4_ERROR getVersionNumber(char *versionString, int size);

/*******************************************************************************
* reset()
* Resets the CMUCam4 forcefully. 
*
* Slow command, forces delay to ensure CMUCam4 has sufficient time to reset. 
* Fails if CMUCam4 does not start up properly.
* Clears internally buffered states for CMUCam4 such as Auto Gain or Auto Pan.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR reset();				

/*******************************************************************************
* startSleepDeeply()
* Puts the CMUcam4 into a very low power state.
*
* Enters SLEEPING state, only responds to wakeup(), all others will fail.
* Clears internally buffered states for CMUCam4 such as Auto Gain or Auto Pan.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure. 
*******************************************************************************/
CMUCAM4_ERROR startSleepDeeply();

/*******************************************************************************
* startSleepLightly()
* Puts the CMUcam4 into a low power state.
*
* Enters SLEEPING state, only responds to wakeup(), all others will fail.
* Clears internally buffered states for CMUCam4 such as Auto Gain or Auto Pan.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR startSleepLightly();	

/*******************************************************************************
* wakeup()
* Wake up the CMUCam4 from low power states.
* 
* Clears the SLEEPING state, allowing other commands to execute again.
* Sends a signal to the CMUCam4 to wake up and resume operations.
* Checks to see if the camera confirms that it has woken up and is idling.
* Default internally buffered states for CMUCam4 such as Auto Gain or Auto Pan.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR wakeup();

/*******************************************************************************
* setCameraBrightness(int brightness)
* Set the camera's current brightness level.
* Default: 0
* Reset when camera restarts (includes sleeping).
*
* brightness: integral value corresponding to brightness. [-255 ~ 255]
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setCameraBrightness(int brightness);

/*******************************************************************************
* setCameraContrast(int contrast)
* Set the camera's current contrast level.
* Default: 0
* Reset when camera restarts (includes sleeping).
*
* contrast: integral value corresponding to contrast [ -31, 31 ]
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setCameraContrast(int contrast);

/*******************************************************************************
* setAutoGain(int active)
* Control Camera Auto Gain.
* Default: Active/on
* Reset when camera restarts (includes sleeping).
*
* active: 0 to turn off, 1 to turn on. 
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setAutoGain(int active);

/*******************************************************************************
* setAutoWhiteBalance(int active)
* Control Camera Auto White Balance.
* Default: Active/On
* Reset when camera restarts (includes sleeping).
*
* active: 0 to turn off, 1 to turn on. 
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setAutoWhiteBalance(int active);

/*******************************************************************************
* setHorizontalMirrorMode(int active)
* Control Camera Horizontal Mirror/Flip mode.
* Default: Not Active/off
* Reset when camera restarts (includes sleeping).
*
* active: 0 to turn off, 1 to turn on. 
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setHorizontalMirrorMode(int active);

/*******************************************************************************
* setVerticalFlipMode(int active)
* Control Camera Vertical Flip/Mirror mode.
* Default: Not Active/off
* Reset when camera restarts (includes sleeping).
*
* active: 0 to turn off, 1 to turn on. 
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setVerticalFlipMode(int active);

/*******************************************************************************
* setBlackWhiteMode(int active)
* Control Camera Black and White mode.
* Default: Not active/Off
* Reset when camera restarts (includes sleeping).
*
* active: 0 to turn off, 1 to turn on. 
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setBlackWhiteMode(int active);

/*******************************************************************************
* setNegativeMode(int active)
* Control Camera Negative Image mode.
* Default: Not Active/off
* Reset when camera restarts (includes sleeping).
*
* active: 0 to turn off, 1 to turn on. 
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setNegativeMode(int active);

/*******************************************************************************
* getButtonState()
* Get Button State as a boolean for if it is currently pressed.
*
* Returns boolean state on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
int getButtonState();

/*******************************************************************************
* getButtonDuration()
* Get Duration for which the button has been in current state. 
*
* Returns duration in current state, CMUCAM4_ERROR on failure.
*******************************************************************************/
int getButtonDuration();

/*******************************************************************************
* getButtonPressed()
* Returns true if the button has been pushed since last call to this function.
*
* Returns boolean if pressed since last call, CMUCAM4_ERROR on failure.
*******************************************************************************/
int getButtonPressed();

/*******************************************************************************
* getButtonReleased()
* Returns true if the button has been released since last call to this function.
*
* Returns boolean if released since last call, CMUCAM4_ERROR on failure.
*******************************************************************************/
int getButtonReleased();

/*******************************************************************************
* panInput()
* Gets the I/O state of the pan pin, true for output mode. 
*
* Returns boolean on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
int panInput();

/*******************************************************************************
* panOutput(int direction, int output)
* Sets the I/O state of the pan pin. 
*
* direction: 0 indicates input, 1 indicates output.
* output: 0 or 1, value only used if setting to output mode.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR panOutput(int direction, int output);

/*******************************************************************************
* tiltInput()
* Gets the I/O state of the tilt pin, true for output mode.
*
* Returns boolean on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
int tiltInput();

/*******************************************************************************
* tiltOutput(int direction, int output)
* Sets the I/O state of the tilt pin. 
*
* direction: 0 indicates input, 1 indicates output.
* output: 0 or 1, value only used if setting to output mode.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR tiltOutput(int direction, int output);

/*******************************************************************************
* getInputs()
* Gets state of pan pin and tilt pin.
* Returns a bit ORd value: bit1 is pan pin, bit0 is tilt pin.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR getInputs();

/*******************************************************************************
* setOutputs(int directions, int outputs)
* Sets the states and values of both pan and tilt pins simultaneously.
*
* directions: 0 for input on both, 3 for output on both (only these values).
* outputs: [0 ~ 3] bit1 sets pan output, bit0 sets tilt output.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
int setOutputs(int directions, int outputs);

/*******************************************************************************
* turnOffLED()
* Turn off the built in LED.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR turnOffLED();

/*******************************************************************************
* turnOnLED(int frequency)
* Turns on the built in LED to blink at a specified frequency.
*
* frequency: the frequency, in hertz, to blink the LED [-1 ~ 10,000,000]
* 				-1 : off, 0 : always on, >0 blinks at specified frequency.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR turnOnLED(int frequency);

/*******************************************************************************
* getServoPosition(int servo)
* Get pulse length information driving the specified servo.
* Returns between [750 ~ 2250] if pin set for output.
*
* servo: address of servo to get info from. [0 ~ 1]
*
* Returns nonnegative value for servo, CMUCAM4_ERROR on failure.
*******************************************************************************/
int getServoPosition(int servo);

/*******************************************************************************
* setServoState(int servo, int active, int pulse)
* Set servo state - Whether it is active and what pulse rate to send it.
*
* servo: address of servo to set. 
* active: Whether the servo should be enabled or not. 
* pulse: What pulse rate, in hertz, to send the servo
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setServoState(int servo, int active, int pulse);

/*******************************************************************************
* autoTiltOn(int isReversed)
* Turn on auto tilt option, specifying whether to set reversed mode.
* Tries to center the camera on the Y coordinate of the tracked centroid.
*
* isReversed: 0 is false, 1 is true.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR autoTilt(int active, int isReversed);

/*******************************************************************************
* autoPanOn(int isReversed)
* Turn on auto tilt option, specifying whether to set reversed mode.
* Tries to center the camera on the X coordinate of the tracked centroid.
*
* isReversed: 0 is false, 1 is true.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR autoPan(int active, int isReversed);

/*******************************************************************************
* setAutoPanParameters(int proportional, int derivative)
* Set PID feedback control options of the auto panning mode.
* Default: P = 400, D = 100
* 
* proportional: Sets the proportional parameter.
* derivative: Sets the derivative parameter.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setAutoPanParameters(int proportional, int derivative);

/*******************************************************************************
* setAutoTiltParameters(int proportional, int derivative)
* Set PID feedback control options of the auto tilting mode.
* Default: P = 400, D = 100
*
* proportional: Sets the proportional parameter.
* derivative: Sets the derivative parameter.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setAutoTiltParameters(int proportional, int derivative);

/*******************************************************************************
* turnOnMonitor()
* Turn on output to external video monitor.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR turnOnMonitor();

/*******************************************************************************
* turnOffMonitor()
* Turn off output to external video monitor.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR turnOffMonitor();

/*******************************************************************************
* freezeMonitor(int active)
* Set freeze mode for output to external monitor.
* Default: Active/on
* Reset when camera restarts.
* active: 0 to allow for monitor updates, 1 to freeze updating.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR freezeMonitor(int active);

/*******************************************************************************
* monitorSignal(int signal)
* Set signal type sent to the external monitor.
* Default: NTSC
* Reset when camera restarts (includes sleeping).
*
* signal: 0 for NTSC, 1 for PAL
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR monitorSignal(int signal);

/*******************************************************************************
* getTrackParameters(CMUCAM4_color_thrsh_t * colors)
* Get tracking parameters in terms of RGB color ranges.
*
* colors: Pointer to struct to fill with tracking colors information.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR getTrackParameters(CMUCAM4_color_range_t * colors);

/*******************************************************************************
* getTrackWindow(CMUCAM4_track_window_t * trkWindow)
* Get tracking window parameters in terms of pixels.
*
* trkWindow: Pointer to struct to fill with tracking window information.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR getTrackWindow(CMUCAM4_track_window_t * trkWindow);

/*******************************************************************************
* setTrackColor(CMUCAM4_color_thrsh_t * colors)
* Set tracking parameters in terms of RGB color ranges.
*
* colors: Pointer to struct of color ranges.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setTrackColor(CMUCAM4_color_range_t * colors);

/*******************************************************************************
* setTrackWindow(CMUCAM4_track_window_t * trkWindow)
* Set tracking window parameters in terms of pixels.
* Restricts which areas of the total display are used in tracking.
*
* trkWindow: Pointer to struct of tracking window dimensions.
*				> Pass NULL to default to full track window size
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setTrackWindow(CMUCAM4_track_window_t * trkWindow);

/*******************************************************************************
* startTrackColor(CMUCAM4_color_thrsh_t * colors)
* Starts streaming color tracking data based on color ranges.
*
* Starts TRACK_COLOR_STREAM state: getTrackData() and stopTracking() valid.
*
* colors: Pointer to struct of color ranges to track.
*           > Can be NULL, and previously stored values will be used.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR startTrackColor(CMUCAM4_color_range_t * colors);

/*******************************************************************************
* startTrackThrsh(CMUCAM4_track_thrsh_t * thrsh)
* Starts streaming color tracking data based on track window points.
*
* Starts TRACK_COLOR_STREAM state: getTrackData() and stopTracking() valid.
*
* thrsh: Pointer to struct of color range thresholds to track.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR startTrackThrsh(CMUCAM4_color_thrsh_t * thrsh);

/*******************************************************************************
* getTrackData(CMUCAM4_track_data_t * data)
* Gets one set of tracking data from tracking stream. 
*
* Blocking read of data, ensures that a full set of data is returned.
* Times out if any one piece of data takes too long to arrive, rather than
*   requiring the entire stream to arrive in the timeout period.
* Ensure this function is called quickly, as it will return data in order
* of it being received, not by what is most recent. 
* Can overflow the serial buffer if this is called too slowly.
*
* Requires TRACK_COLOR_STREAM state to run.
*
* data: pointer to a struct to fill with tracking data.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR getTrackData(CMUCAM4_track_data_t * data);

/*******************************************************************************
* stopTracking()
* Stops streaming tracking data.
* 
* Clears the TRACK_COLOR_STREAM state, allowing other commands to be called.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR stopTracking();

/*******************************************************************************
* pollTrackColor(CMUCAM4_color_thrsh_t * colors, CMUCAM4_track_data_t * data)
* Loose wrapper to start, read and stop the track color data stream.
* Set tracking parameters in terms of RGB color ranges.
* 
* Blocking read of data, ensures that a full set of data is returned.
* Times out if any one piece of data takes too long to arrive, rather than
*   requiring the entire stream to arrive in the timeout period.
*
* colors: Pointer to struct of color ranges.
* data: pointer to a struct to fill with tracking data.
* 
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR pollTrackColor(CMUCAM4_color_range_t * colors, 
								CMUCAM4_track_data_t * data);

/*******************************************************************************
* pollTrackColor(CMUCAM4_color_thrsh_t * colors, CMUCAM4_track_data_t * data)
* Loose wrapper to start, read and stop the track color data stream.
* Set tracking parameters in terms of RGB color ranges.
* 
* Blocking read of data, ensures that a full set of data is returned.
* Times out if any one piece of data takes too long to arrive, rather than
*   requiring the entire stream to arrive in the timeout period.
*
* colors: Pointer to struct of color ranges.
* data: pointer to a struct to fill with tracking data.
* 
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR pollTrackThrsh(CMUCAM4_color_thrsh_t * colors, 
								CMUCAM4_track_data_t * data);

/*******************************************************************************
* startHistogram(int color, int numBins)
* Starts streaming color histogram data.
* 
* Sets the HISTOGRAM_STREAM state: getHistogramData and stopHistogram valid.
* Takes next lowest power of two bins (truncating log_2) 
*
* color: 0 for Red, 1 for Green, 2 for Blue
* numBins: How many partitions to divide colors into. 
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR startHistogram(int color, int numBins);

/*******************************************************************************
* getHistogramData(unsigned char * bins, int numBins)
* Gets one set of histogram data from histogram stream. 
* Influenced by setTrackWindow and setColorTracking.
*
* Blocking read of data, ensures that a full set of data is returned.
* Times out if any one piece of data takes too long to arrive, rather than
*   requiring the entire stream to arrive in the timeout period.
* Ensure this function is called quickly, as it will return data in order
* of it being received, not by what is most recent. 
* Can overflow the serial buffer if this is called too slowly.
*
* Requires HISTOGRAM_STREAM state to run.
*
* bins: unsigned character array to fill with histogram data.
* numBins: Tells the number of bins there are to fill, will fill only as many
*           > as the CMUCam4 returns, ensured to be no more than numBins passed 
*           > to startHistogram. SHOULD BE A POWER OF TWO. It will round down 
* 			> to the nearest power of two.
* 
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR getHistogramData(unsigned char * bins, int numBins);

/*******************************************************************************
* stopHistogram()
* Stops streaming histogram data.
* 
* Clears the HISTOGRAM_STREAM state, allowing other commands to be called.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR stopHistogram();

/*******************************************************************************
* pollHistogram(unsigned char * bins, int numBins)
* Loose wrapper to start, read and stop a histogram data stream. 
* Influenced by setTrackWindow and setColorTracking.
*
* Blocking read of data, ensures that a full set of data is returned.
* Times out if any one piece of data takes too long to arrive, rather than
*   requiring the entire stream to arrive in the timeout period.
*
* color: channel color to get, 0 : red, 1 : green, 2 : blue.
* bins: unsigned character array to fill with histogram data.
* numBins: tells the number of bins there are to fill, will fill only as many
*           > as the CMUCam4 returns (power of two). SHOULD BE A POWER OF TWO.
* 
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR pollHistogram(int color, unsigned char * bins, int numBins);

/*******************************************************************************
* startMean()
* Starts streaming color statistics data.
* 
* Sets the GET_MEAN_STREAM state: getMean and stopMean valid.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR startMean();

/*******************************************************************************
* getMean(CMUCAM4_mean_data_t * data)
* Gets one set of color statistics from mean stream. 
*
* Blocking read of data, ensures that a full set of data is returned.
* Times out if any one piece of data takes too long to arrive, rather than
*   requiring the entire stream to arrive in the timeout period.
* Ensure this function is called quickly, as it will return data in order
* of it being received, not by what is most recent. 
* Can overflow the serial buffer if this is called too slowly.
*
* Requires GET_MEAN_STREAM state to run.
*
* data: pointer to struct to fill with color statistics data.
* 
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR getMean(CMUCAM4_mean_data_t * data);

/*******************************************************************************
* stopMean()
* Stops streaming color statistics data.
* 
* Clears the GET_MEAN state, allowing other commands to be called.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR stopMean();

/*******************************************************************************
* pollMean(CMUCAM4_mean_data_t * data)
* Loose wrapper to start, read and stop the color statistics stream
* Gets one set of color statistics from mean stream. 
*
* Blocking read of data, ensures that a full set of data is returned.
* Times out if any one piece of data takes too long to arrive, rather than
*   requiring the entire stream to arrive in the timeout period.
*
* data: pointer to struct to fill with color statistics data.
* 
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR pollMean(CMUCAM4_mean_data_t * data);

/*******************************************************************************
* startTestMode(int active)
* Set test mode for CMUCam4.
* Default: Not active/off
* Reset when camera restarts (including sleeping).
*
* active: 0 to turn off, 1 to turn on.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR startTestMode(int active);

/*******************************************************************************
* setColorTracking(int active)
* Set color tracking mode in terms of YUV and RGB modes
* Default: RGB
* Reset when camera restarts (includes sleeping).
*
* active: 0 to turn on RGB, 1 to turn on YUV.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setColorTracking(int active);

/*******************************************************************************
* setHistogramTracking(int active)
* Set histogram data to calculate on tracked data only.
* Default: Not Active/off
*
* active: 0 to turn off, 1 to turn on.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setHistogramTracking(int active);

/*******************************************************************************
* setInvertedFilterMode(int active)
* Set inverted filtering mode, track all colors outside threshholds instead.
* Default: Not Active/off
* Reset when camera restarts (includes sleeping).
*
* active: 0 to turn off, 1 to turn on.
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setInvertedFilterMode(int active);

/*******************************************************************************
* setNoiseFilterMode(int active)
* Set noise filtering mode.
* Default: Not Active/off
* Reset when camera restarts (includes sleeping).
*
* active: 0 to turn off, [1 ~ 255] to set threshold
*
* Returns CMUCAM4_RETURN_SUCCESS on success, CMUCAM4_ERROR on failure.
*******************************************************************************/
CMUCAM4_ERROR setNoiseFilterMode(int thresh); 


/*******************************************************************************
********************************************************************************
******   PRIVATE
********************************************************************************
*******************************************************************************/
private:

/*******************************************************************************
* Stores the current state of the CMUCam4.
*******************************************************************************/
char _state;

/*******************************************************************************
* Sets state variable if and only if the state is void. 
*******************************************************************************/
int _setState(int state);

/*******************************************************************************
* Unsets the current state if and only if it is set, otherwise throws an error.
* Void state is except and will not raise an error if set. 
*******************************************************************************/
int _unsetState(int state);

/*******************************************************************************
* Gets the integer value of the current state variable.
*******************************************************************************/
int _getState();

/*******************************************************************************
* Byte array containing the bytes to write to the serial buffer.
*******************************************************************************/
char _commandBuffer[CMUCAM4_MAX_COMMAND_SIZE];

/*******************************************************************************
* Sends a command to the CMUCam4 through the serial port.
*
* Flushes out the serial buffer before sending the command.
* Encapsulates string terminating characters
* Commands follow strict formatting conventions from CMUCam4
* 
* Argument 1: A string containing a serial command
* 
* Returns 0 on success and -1 on failure.
*******************************************************************************/
int _sendCommand(const char * command);		

/*******************************************************************************
* Reads a single character from the serial buffer. 
* Checks for timeout for reading only one character.
* 
* Throws Timeout errors.
*******************************************************************************/
char _readChar();

/*******************************************************************************
* Reads a single integer from the serial buffer.
* If delim is non null it will return the character found after the integer.
*
* Throws Timeout errors.
*******************************************************************************/
int _readInt(char * delim);

/*******************************************************************************
* Reads from the CMUCam4's serial buffer until a line starting "ACK" is read.
* 
* Calls _serialRead() until the sequence is found.
* Throws Timeout, Flow and Error errors as defined in _serialRead().
*******************************************************************************/
int _waitACK();

/*******************************************************************************
* Reads from the CMUCAM4's serial buffer until a sequence ":<endline>" is read.
* 
* Calls _serialRead() until the sequence is found.
* Throws Timeout, Flow and Error errors as defined in _serialRead()
*******************************************************************************/
int _waitIdle();


/*******************************************************************************
* Byte array containing the bytes read from the serial buffer
*******************************************************************************/
char _byteBuffer[CMUCAM4_SERIAL_SIZE];

/*******************************************************************************
* Number of bytes currently read into the _byteBuffer array 
*
* Does not represent the size of the buffer, only reads till full or new line
*******************************************************************************/
int _byteCount;		


/*******************************************************************************
* Reads one line of bytes from the CMUCam4 serial port. 
*
* Catches timeout and overflow errors. 
* Timeout when no end of line character is found in timeout period.
* Overflow when buffer fills up before reading an end of line character.
* Expects all lines to end with the correct end of line character.
* Will correctly catch idle lines (which have no end of line character).
*
* Returns an CMUCAM4_ERROR error message value
*******************************************************************************/
CMUCAM4_ERROR _serialRead();				
			
/*******************************************************************************
* Required global buffer for setjmp and longjmp, do not directly modify!
*******************************************************************************/
jmp_buf _longJumpEnv;


};

#endif

/*******************************************************************************
*                     TERMS OF USE: MIT License
********************************************************************************
* Permission is hereby granted, free of charge, to any person obtaining a copy 
* of this software and associated documentation files (the "Software"), to deal 
* in the Software without restriction, including without limitation the rights 
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
* copies of the Software, and to permit persons to whom the Software is 
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in 
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*******************************************************************************/
